// (c) 1999 - 2025 OneSpan North America Inc. All rights reserved.


/////////////////////////////////////////////////////////////////////////////
//
//
// This file is example source code. It is provided for your information and
// assistance. See your licence agreement for details and the terms and
// conditions of the licence which governs the use of the source code. By using
// such source code you will be accepting these terms and conditions. If you do
// not wish to accept these terms and conditions, DO NOT OPEN THE FILE OR USE
// THE SOURCE CODE.
//
// Note that there is NO WARRANTY.
//
//////////////////////////////////////////////////////////////////////////////


import MSSOrchestration

class RemoteViewController: UIViewController {
    @IBOutlet weak var imageView: UIView!
    @IBOutlet weak var buttonsView: UIView!
    @IBOutlet weak var titleLabel: UILabel!
    @IBOutlet weak var subtitleLabel: UILabel!
    @IBOutlet weak var noButton: UIButton!
    @IBOutlet weak var yesButton: UIButton!
    
    private let userAuthenticationDelegate = UserAuthenticationViewController()
    private let serverCallHandler = ServerCallHandler()
    private let orchestrationDelegate = OrchestrationSampleDelegate()
    
    private var orchestrator: Orchestrator?
    private var progressDialog: UIView?
    private var remoteAuthenticationDisplayDataCaller: DisplayDataCaller?
    private var remoteTransactionDisplayDataCaller: DisplayDataCaller?
    private var command = ""
    
    override func viewDidLoad() {
        super.viewDidLoad()
        setupView()
        setupOrchestrator()
        
        // Open remote command
        orchestrator?.execute(command: command)
    }
    
    override func viewWillAppear(_ animated: Bool) {
        super.viewWillAppear(animated)
        displayProgress(message: "dialog_progress_remote_auth".localized)
    }
    
    // MARK: Setup
    private func setupView() {
        title = "title_activity_remote_validation".localized
        navigationItem.hidesBackButton = false
        
        let buttonRadius = 3
        imageView.layer.cornerRadius = imageView.frame.size.width / 2
        imageView.clipsToBounds = true
        subtitleLabel.text = "hi_subtitle".localized
        noButton.setTitle("btn_no".localized.uppercased(), for: UIControl.State.normal)
        noButton.layer.cornerRadius = CGFloat(buttonRadius)
        yesButton.setTitle("btn_yes".localized.uppercased(), for: UIControl.State.normal)
        yesButton.layer.cornerRadius = CGFloat(buttonRadius)
        buttonsView.isHidden = true
        
        navigationItem.leftBarButtonItem = UIBarButtonItem(barButtonSystemItem: UIBarButtonItem.SystemItem.cancel,
                                                           target: self,
                                                           action: #selector(backToMain))
    }
    
    private func setupOrchestrator() {
        orchestrationDelegate.progressDialog = progressDialog
        orchestrationDelegate.viewController = self
        orchestrator = OrchestratorUtils.getOrchestrator(delegate: orchestrationDelegate)
        
        guard let orchestrator = orchestrator else {
            assertionFailure("Can't get orchestrator")
            return
        }
        
        // Used for custom password instead of default one
        userAuthenticationDelegate.useExternalPassword(orchestrator: orchestrator, viewController: self)
        
        orchestrator.set(remoteAuthenticationDelegate: self)
        orchestrator.set(remoteTransactionDelegate: self)
        
        serverCallHandler.viewController = self
        serverCallHandler.orchestrator = orchestrator
    }
    
    // MARK: setter
    func setCommand(_ command: String) {
        print("remote command: ", command)
        self.command = command
    }
    
    // MARK: IBAction
    @IBAction func onAcceptButtonClick(_ sender: Any) {
        acceptLoginRequest()
    }
    
    @IBAction func onRejectButtonClick(_ sender: Any) {
        rejectLoginRequest()
    }
    
    // MARK: Private RemoteAuth
    private func acceptLoginRequest() {
        // Hide accept/reject buttons
        buttonsView.isHidden = true
        subtitleLabel.text = "hi_subtitle".localized
        
        if remoteAuthenticationDisplayDataCaller != nil {
            // Display progress dialog and accept the login request
            displayProgress(message: "dialog_progress_remote_auth".localized)
            remoteAuthenticationDisplayDataCaller?.remoteDataApproved()
        } else if remoteTransactionDisplayDataCaller != nil {
            // Display progress dialog and accept the transaction request
            displayProgress(message: "dialog_progress_remote_transaction".localized)
            remoteTransactionDisplayDataCaller?.remoteDataApproved()
        }
    }
    
    private func rejectLoginRequest() {
        // Hide accept/reject buttons
        buttonsView.isHidden = true
        subtitleLabel.text = "hi_subtitle".localized
        
        if remoteAuthenticationDisplayDataCaller != nil {
            // Display progress dialog and reject the login request
            displayProgress(message: "dialog_progress_remote_auth".localized)
            remoteAuthenticationDisplayDataCaller?.remoteDataRejected()
        } else if remoteTransactionDisplayDataCaller != nil {
            // Display progress dialog and reject the transaction request
            displayProgress(message: "dialog_progress_remote_transaction".localized)
            remoteTransactionDisplayDataCaller?.remoteDataRejected()
        }
    }
    
    // MARK: Pasword error
    private func onPasswordError(_ error: OrchestrationError) {
        // Hide progress dialog and display failure alert
        UIUtils.hideProgress(progressDialog)
        UIUtils.displayAlert(for: error, on: self)
    }
    
    // MARK: Private utils
    private func displayExitDiplayAlert(title: String, message: String) {
        // Hide progress dialog and display failure alert
        UIUtils.hideProgress(progressDialog)
        UIUtils.displayAlert(controller: self, title: title, message: message) { [weak self] in
            self?.backToMain()
        }
    }
    
    @objc private func backToMain() {
        navigationController?.popToRootViewController(animated: true)
    }
    
    private func displayProgress(message: String) {
        if progressDialog != nil {
            UIUtils.hideProgress(progressDialog)
        }
        
        progressDialog = UIUtils.displayProgress(controller: self, message: message)
        orchestrationDelegate.progressDialog = progressDialog
    }
    
    private func sendCommandToServer(_ command: String) {
        serverCallHandler.progressDialog = progressDialog
        serverCallHandler.sendCommandToServer(command) {
            self.backToMain()
        }
    }
    
    private func parseRemoteTransaction(dataToDisplay: String) -> String {
        return dataToDisplay.components(separatedBy: "#").joined(separator: "\n")
    }
}

extension RemoteViewController: RemoteTransactionDelegate {
    func orchestrator(_ orchestrator: Orchestrator, user: OrchestrationUser, remoteTransactionResponseNeededFor dataToDisplay: String, responseCaller: DisplayDataCaller) {
        // Hide progress dialog
        UIUtils.hideProgress(progressDialog)
        
        // Keep reference on caller for later
        remoteTransactionDisplayDataCaller = responseCaller
        let dataToSign = parseRemoteTransaction(dataToDisplay: dataToDisplay)
        var labelText = String(format: "want_to_sign_data".localized, dataToSign)
        let userString = String(format: "user_to_authenticate".localized, user.identifier)
        labelText.append(userString)
        subtitleLabel.text = labelText
        
        // Show accept/reject buttons
        buttonsView.isHidden = false
    }
    
    func orchestrator(_ orchestrator: Orchestrator, didFinishRemoteTransactionStepWith command: String) {
        // Send command to server
        sendCommandToServer(command)
    }
    
    func orchestrator(_ orchestrator: Orchestrator, didFinishRemoteTransactionFlowWith state: RemoteSessionState) {
        let message: String
        switch state {
        case .accepted:
            message = "dialog_content_remote_transaction_success_validated".localized
        case .refused:
            message = "dialog_content_remote_transaction_success_rejected".localized
        default:
            fatalError("unknown state")
        }
        
        displayExitDiplayAlert(title: "dialog_title_remote_transaction".localized, message: message)
    }
    
    func orchestrator(_ orchestrator: Orchestrator, didOutdateRemoteTransactionSessionWith reason: RemoteSessionOutdatedReason) {
        var message = ""
        switch reason {
        case .sessionExpired:
            message = "dialog_content_remote_session_expired".localized
        case .sessionAccepted:
            message = "dialog_content_remote_session_accepted".localized
        case .sessionRefused:
            message = "dialog_content_remote_session_refused".localized
        case .sessionUnknown:
            message = "dialog_content_remote_session_unknown".localized
        default:
            fatalError("unknown state")
        }
        
        displayExitDiplayAlert(title: "dialog_title_remote_session_outdated".localized, message: message)
    }
    
    func orchestratorDidAbortRemoteTransaction(_ orchestrator: Orchestrator) {
        displayExitDiplayAlert(title: "dialog_title_remote_transaction".localized, message: "dialog_content_remote_transaction_abortion".localized)
    }
    
    func orchestrator(_ orchestrator: Orchestrator, didReceiveRemoteTransactionError error: OrchestrationError) {
        // Show password error
        onPasswordError(error)
    }
}

extension RemoteViewController: RemoteAuthenticationDelegate {
    func orchestrator(_ orchestrator: Orchestrator, user: OrchestrationUser, remoteAuthenticationResponseNeededFor dataToDisplay: String, responseCaller: DisplayDataCaller) {
        // Hide progress dialog
        UIUtils.hideProgress(progressDialog)
        
        // Keep reference on caller for later
        remoteAuthenticationDisplayDataCaller = responseCaller
                
        // Parse data and display the host on which the user wants to connect
        var subtitle = ""
        var host = ""
        if dataToDisplay.contains("#") {
            let split = dataToDisplay.components(separatedBy: "#")
            let user = split.count >= 1 ? split[0]: ""
            titleLabel.text = String(format: "hi".localized, user)
            
            host = String(format: "sign_to".localized, split.count >= 2 ? split[1]: "")
        }
        
        subtitle = String(format: "want_to_sign_in".localized, host)
        let userString = String(format: "user_to_authenticate".localized, user.identifier)
        subtitle.append(userString)
        subtitleLabel.text = subtitle
        
        // Show accept/reject buttons
        buttonsView.isHidden = false
    }
    
    func orchestrator(_ orchestrator: Orchestrator, didFinishRemoteAuthenticationStepWith command: String) {
        // Send command to server
        sendCommandToServer(command)
    }
    
    func orchestrator(_ orchestrator: Orchestrator, didFinishRemoteAuthenticationFlowWith state: RemoteSessionState) {
        let message: String
        switch state {
        case .accepted:
            message = "dialog_content_remote_auth_success_signed_in".localized
        case .refused:
            message = "dialog_content_remote_auth_success_rejected".localized
        case .unsupported(let rawValue):
            assertionFailure("unsupported: \(rawValue)")
            message = "dialog_content_remote_auth_success_unsupported".localized.replacingOccurrences(of: "%@", with: "\(rawValue)")
        @unknown default:
            fatalError("unknown state")
        }
        
        displayExitDiplayAlert(title: "dialog_title_remote_auth".localized, message: message)
    }
    
    func orchestrator(_ orchestrator: Orchestrator, didOutdateRemoteAuthenticationSessionWith reason: RemoteSessionOutdatedReason) {
        var message: String
        switch reason {
        case .sessionExpired:
            message = "dialog_content_remote_session_expired".localized
        case .sessionAccepted:
            message = "dialog_content_remote_session_accepted".localized
        case .sessionRefused:
            message = "dialog_content_remote_session_refused".localized
        case .sessionUnknown:
            message = "dialog_content_remote_session_unknown".localized
        default:
            message = ""
        }
        
        displayExitDiplayAlert(title: "dialog_title_remote_session_outdated".localized, message: message)
    }
    
    func orchestratorDidAbortRemoteAuthentication(_ orchestrator: Orchestrator) {
        displayExitDiplayAlert(title: "dialog_title_remote_auth".localized, message: "dialog_content_remote_auth_abortion".localized)
    }
    
    func orchestrator(_ orchestrator: Orchestrator, didReceiveRemoteAuthenticationError error: OrchestrationError) {
        // Show password error
        onPasswordError(error)
    }
}
